package org.apache.jackrabbit.ocm;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Repository;
import javax.jcr.Session;

import org.apache.jackrabbit.ocm.exception.RepositoryException;
import org.apache.jackrabbit.ocm.manager.ObjectContentManager;
import org.apache.jackrabbit.ocm.manager.impl.ObjectContentManagerImpl;
import org.apache.jackrabbit.ocm.mapper.Mapper;
import org.apache.jackrabbit.ocm.mapper.impl.annotation.AnnotationMapperImpl;
import org.apache.jackrabbit.ocm.model.Author;
import org.apache.jackrabbit.ocm.model.PressRelease;
import org.apache.jackrabbit.ocm.model.Url;
import org.apache.jackrabbit.ocm.util.RepositoryUtil;

public class Main {

	public void startTutorial()
	{
		try
		{
			System.out.println("Start the tutorial ...");
			ObjectContentManager ocm = this.getOCM();

			// Insert a PressRelease
			System.out.println("Insert a press release in the repository");
			PressRelease pressRelease = new PressRelease();
			pressRelease.setPath("/newtutorial");
			pressRelease.setTitle("This is the first tutorial on OCM");
			pressRelease.setPubDate(new Date());
			pressRelease.setContent("Many Jackrabbit users asked to the dev team to make a tutorial on OCM");
			pressRelease.setAuthor(new Author("Christophe", "Lombart"));

			List<Url> urls =  new ArrayList<Url>();
			urls.add(new Url("http://www.apache.org", "ASF web site", "A nice open source fondation"));
			urls.add(new Url("http://jackrabbit.apache.org", "Jackrabbit", "A nice JCR implementation"));
			pressRelease.setUrls(urls);

			// an alternative to a collection, .... 
			Map<String, Url> map = new HashMap<String, Url>();
			map.put("Apache", new Url("http://www.apache.org", "ASF web site", "A nice open source fondation"));
			map.put("Jackrabbit", new Url("http://jackrabbit.apache.org", "Jackrabbit", "A nice JCR implementation"));
			pressRelease.setMap(map);

			ocm.insert(pressRelease);
			ocm.save();

			// Retrieve
			System.out.println("Retrieve a press release from the repository");
			pressRelease = (PressRelease) ocm.getObject("/newtutorial");
			System.out.println("PressRelease title : " + pressRelease.getTitle());
			System.out.println("PressRelease author : " + pressRelease.getAuthor().getFirstName()
					                                    + " " + pressRelease.getAuthor().getLastName());
			urls = pressRelease.getUrls();
			System.out.println("******  URLS (Collection) : ");
			for (Url url : urls) {
				System.out.println("URL : " + url.getUrl());
			}

			System.out.println("******  URLS (Map) : ");
			for (String key : map.keySet())
			{
				System.out.println("URL : " + key + " - " + map.get(key).getUrl());
			}
			
			
			// Delete
			System.out.println("Remove a press release from the repository");
			ocm.remove(pressRelease);
			ocm.save();


		}
		catch (Exception e)
		{
			throw new RepositoryException(e);
		}

	}



	private Session getLocalSession() {


		Repository repository = RepositoryUtil.getTrancientRepository();
		Session session = RepositoryUtil.login(repository, "username", "superuser");

		return session;
	}

	private ObjectContentManager getOCM()
	{
		// Get a JCR session
		Session session = getLocalSession();

		// Register the persistent classes
		List<Class> classes = new ArrayList<Class>();
		classes.add(PressRelease.class);
		classes.add(Author.class);
		classes.add(Url.class);

		Mapper mapper = new AnnotationMapperImpl(classes);
		return new ObjectContentManagerImpl(session, mapper);
	}


	/**
	 * @param args
	 */
	public static void main(String[] args)
	{
		Main main = new Main();
		main.startTutorial();
		
	}

}
